/* 
 *  InteractionTimer.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;

public abstract class InteractionTimer {

    private static final Logger LOG = Logger.getLogger(InteractionTimer.class);
    private int timerLength = 0;
    private int currentTime = 0;
    private final Timer timer;
    private final AtomicBoolean stopping = new AtomicBoolean(false);
    private final String interactionID;
    private final InteractionDataGUI gui;

    public InteractionTimer(int timerLength, String name, String interactionID, InteractionDataGUI gui) {
        this.timerLength = timerLength;
        this.currentTime = timerLength;
        this.interactionID = interactionID;
        this.gui = gui;
        timer = new Timer(name + "-" + interactionID);
        // Start the timer with a small delay to ensure the interaction list has been displayed before the first timer tick
        timer.scheduleAtFixedRate(new InteractionTimerTask(), 100, 1000);
    }

    public boolean restartTimer() {
        String methodName = "restartTimer() ";
        synchronized (this) {
            if (!stopping.equals(true)) {
                LOG.trace(methodName + " restarting timer");
                currentTime = timerLength;
                return true;
            } else {
                LOG.trace(methodName + " timer already stopping, don't restart");
                return false;
            }
        }
    }

    public boolean extendTimer() {
        String methodName = "extendTimer() ";
        synchronized (this) {
            if (!stopping.equals(true)) {
                LOG.trace(methodName + " extending timer");
                currentTime += timerLength;
                return true;
            } else {
                LOG.trace(methodName + " timer already stopping, don't extend");
                return false;
            }
        }
    }
    
    public void stopTimer() {
        String methodName = "stopTimer() ";
        LOG.trace(methodName);
        synchronized (this) {
            timer.cancel();
            stopping.set(true);
        }
    }

    /**
     * @return the interactionID
     */
    public String getInteractionID() {
        return interactionID;
    }

    private class InteractionTimerTask extends TimerTask {

        @Override
        public void run() {
            String methodName = "run() ";
            LOG.trace(methodName + " currentTime:" + currentTime);
            synchronized (this) {             
                if (!stopping.equals(true)) {
                    InteractionData interaction = (InteractionData) gui.getItem(getInteractionID());
                    interaction.setTimeLeft(currentTime);
                    gui.UpdateInteractionData(interaction);
                    if (currentTime <= 0) {
                        LOG.trace(methodName + " timer expired");
                        executeExpiryAction();
                        timer.cancel();
                        stopping.set(true);
                    }
                }
                currentTime--;
            }
        }

    }

    public abstract void executeExpiryAction();
}
